
// AUTHOR: Marco Marchetti, Bruce A. Edgar Lab, Huntsman Cancer Institute, Salt Lake City, 84112, UT, USA

// DESCRIPTION: Macro designed for the manual tracking of cells in two-channel time-lapse images of adult Drosophila midguts. Images must be
// two-channel time-lapse z-stacks, with channel 1 = His2Av.mRFP and channel 2 = esgTS > nlsGFP. 

// MAIN

// Opening a time-lapse image and extracting file name and path
open();
title = getTitle();
path = getDirectory("image");

// Determining upscale factor for sub-pixel nuclear area calculation
upscale = getUpscaleFactor();

// User inputs the voxel size
defineVoxelSize();
getVoxelSize(width, height, depth, unit);

// Defining pixel size of 50um window to be used around the cell of interest for local background estimation
window = 25 / width;

// Manual tracking by drawing the outline of a cell and its progeny across all time-points, adding each outline to the ROI Manager
setTool("freehand");
run("Set Measurements...", "area mean min centroid fit shape integrated median stack redirect=None decimal=3");
run("ROI Manager...");
roiManager("reset");
waitForUser("Manual Tracking", "Follow a cell and its progeny\nthrought time, drawing their\noutline and adding it to the\nROI Manager, then press OK.\nN.B. ROI order does not matter.");

// Saving Regions of Interest (ROIs)
out_name = path + substring(title, 0, lengthOf(title) - 4);
roiManager("deselect");
roiManager("save", out_name + ".zip");

// Analyze the ROIs to extract centroid coordinates, shape descriptors, nuclear area, and fluorescence
print("T\tX\tY\tZ\tArea\tMinAxis\tMaxAxis\tCircularity\tRFP-BG\tMeanRFP\tNormMeanRFP\tMedianRFP\tMinRFP\tMaxRFP\tRawRFP\tGFP-BG\tMeanGFP\tNormMeanGFP\tMedianGFP\tMinGFP\tMaxGFP\tRawGFP");
for(r=0; r<roiManager("count"); r++) {
	measureCell();
}

// Closing windows and saving results
selectWindow("ROI Manager");
run("Close");
selectWindow("Log");
saveAs("txt", out_name + ".txt");
run("Close");
close();

/// FUNCTIONS

function getUpscaleFactor() {
	
	// Determining the minimum upscaling factor to push image to >= 8000 px along the largest axis.
	// This will make cell area calculations more precise
	getDimensions(img_width, img_height, img_channels, img_slices, img_frames);
	original_dimension = maxOf(img_width, img_height);
	upscaled_dimension = original_dimension;
	upscale_factor = 1;
	while(upscaled_dimension < 8000) {
		upscale_factor += 1;
		upscaled_dimension = original_dimension * upscale_factor;
	}
	return upscale_factor;
	
}

function defineVoxelSize() {
	
	// User inputs the voxel size
	getVoxelSize(width, height, depth, unit);
	Dialog.create("Voxel Size");
	Dialog.addMessage("Please set/confirm scale (microns / px).");
	Dialog.addNumber("Voxel width:", width);
	Dialog.addNumber("Voxel height:", height);
	Dialog.addNumber("Voxel depth:", depth);
	Dialog.show();
	width = Dialog.getNumber();
	height = Dialog.getNumber();
	depth = Dialog.getNumber();
	setVoxelSize(width, height, depth, "microns");
	
}

function measureCell() {
	
	// Selecting ROI and extracting frame and z info
	roiManager("select", r);
	Stack.getPosition(channel, z, frame);
	z = (z - 1) * depth; // Adjusting z position based on z-step
	run("Select None");
	
	// Upscaling image to make nuclear area and centroid calculations more accurate
	run("Duplicate...", "title=Temp channels=1 slices=4");
	run("Scale...", "x=" + upscale + " y=" + upscale + " interpolation=Bilinear average create title=Enlarged"); // Upscaling image
	roiManager("select", r);
	run("Scale... ", "x=" + upscale + " y=" + upscale); // Scaling selection

	// Measuring nuclear area, coordinates, and shape descriptors
	run("Measure");
	x = getResult("X", 0);
	y = getResult("Y", 0);
	area = getResult("Area", 0);
	min_axis = getResult("Minor", 0);
	max_axis = getResult("Major", 0);
	circularity = getResult("Circ.", 0);
	selectWindow("Temp");
	close();
	selectWindow("Enlarged");
	close();
	
	// Measuring RFP
	roiManager("select", r);
	Stack.setChannel(1);
	run("Measure");
	mean_rfp = getResult("Mean", 1);
	median_rfp = getResult("Median", 1);
	min_rfp = getResult("Min", 1);
	max_rfp = getResult("Max", 1);
	raw_rfp = getResult("RawIntDen", 1);
	
	// Measuring RFP background
	run("Select None");
	makeRectangle(x / width - window, y / height - window, 2 * window, 2 * window);
	run("Measure");
	bg_rfp = getResult("Mean", 2);
	norm_rfp = mean_rfp / bg_rfp;
	
	// Measuring GFP
	roiManager("select", r);
	Stack.setChannel(2);
	run("Measure");
	mean_gfp = getResult("Mean", 3);
	median_gfp = getResult("Median", 3);
	min_gfp = getResult("Min", 3);
	max_gfp = getResult("Max", 3);
	raw_gfp = getResult("RawIntDen", 3);
	
	// Measuring GFP local background
	run("Select None");
	makeRectangle(x / width - window, y / height - window, 2 * window, 2 * window);
	run("Measure");
	bg_gfp = getResult("Mean", 4);
	norm_gfp = mean_gfp / bg_gfp;
	
	// Writing measurements
	print(""+frame+"\t"+x+"\t"+y+"\t"+z+"\t"+area+"\t"+min_axis+"\t"+max_axis+"\t"+circularity+"\t"+bg_rfp+"\t"+mean_rfp+"\t"+norm_rfp+"\t"+median_rfp+"\t"+min_rfp+"\t"+max_rfp+"\t"+raw_rfp+"\t"+bg_gfp+"\t"+mean_gfp+"\t"+norm_gfp+"\t"+median_gfp+"\t"+min_gfp+"\t"+max_gfp+"\t"+raw_gfp);
	selectWindow("Results");
	run("Close");
	
}